<?php

require_once 'lib/opcodes.php';
require_once 'lib/types.php';

class Machine {
    private Components $components;
    private Processor $processor;
    private array $interruptHandlers = [];

    public function __construct() {
        $this->components = new Components($depth = 4);
        $this->processor = new Processor($depth, $this->components);
    }

    public function run() : void {
        while(true) {
            $this->processor->tick();
            foreach($this->interruptHandlers as &$handler) {
                $handler();
            }
        }
    }
    public function attachInterrupt(Closure $handler) : void {
        $this->interruptHandlers[] = $handler;
    }
}

class Debugger {
    public function __construct() {
        if(($socket = socket_create(AF_INET, SOCK_STREAM, SOL_TCP)) === false)
            throw new Exception("Debugger: unable to create socket: " . socket_strerror(socket_last_error($socket)));
        if(socket_bind($socket, '127.0.0.1', 41943) === false)
            throw new Exception("Debugger: unable to bind socket: " . socket_strerror(socket_last_error($socket)));
        if(socket_listen($socket, 1) === false)
            throw new Exception("Debugger: unable to listen on socket: " . socket_strerror(socket_last_error($socket)));

        echo "Connect to 127.0.0.1:41943 via raw socket software to continue.\n";
        while(($clientSocket = socket_accept($socket)) === false) { sleep(1); }
        socket_write($clientSocket, "Hi there!\n");
    }
}

class Components {
    private array $components;

    private function readFile($filename) : array {
        $componentList = [];
        foreach(file($filename, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES) as &$componentString) {
            $array2push = explode(' ', $componentString, 2);
            if(count($array2push) === 1) $array2push[] = '';
            $componentList[] = $array2push;
        }
        return $componentList;
    }

    public function __construct(int $depth) {
        $mainPool = [];
        $endPool = [];
        $currentPool = 'mainPool';
        $offset = 0;
        $isBusrouterMarked = false;
        $mainPoolSize = 0;
        $endPoolSize = 0;
        $busrouterSize = 1;
        foreach($this->readFile("components.txt") as &$component) {
            if($isBusrouterMarked) throw new Exception("busrouter must be at the end of the address space");
            if($component[0] === "empty") {
                if($currentPool === 'endPool')
                    throw new Exception("Only one instance of empty allowed");
                $currentPool = 'endPool';
                continue;
            }
            if($component[0] === "busrouter") {
                if($currentPool === 'mainPool')
                    throw new Exception("busrouter must be at the end of the address space");
                $endPoolSize += $busrouterSize;
                $isBusrouterMarked = true;
                continue;
            }

            try {
                require_once 'components/' . $component[0] . '.php';
                $component2load = new LoadedComponent(new $component[0]($component[1]));
                if($currentPool === 'mainPool') {
                    $component2load->offset = $offset;
                    $offset += $component2load->getBytes();
                    $mainPoolSize += $component2load->getBytes();
                } else {
                    $endPoolSize += $component2load->getBytes();
                }
                $busrouterSize += $depth * 3 + strlen($component[0]) + 1; // offset + size + metadataptr + name + nul
                $$currentPool[] = $component2load;
                if($mainPoolSize + $endPoolSize + $busrouterSize > 2 ** ($depth * 8) - 1) throw new Exception("Not enough space in address space");
            } catch(Throwable $error) {
                echo "Can't load component " . $component[0] . ": $error";
                die();
            }
        }
        $endPoolOffset = (2 ** ($depth * 8)) - $endPoolSize;
        foreach($endPool as &$component) {
            $component->offset = $endPoolOffset;
            $endPoolOffset += $component->getBytes();
        }
        $components = array_merge($mainPool, $endPool);
        if($isBusrouterMarked) {
            $busrouter = new LoadedComponent(new busrouter($depth, $components, (2 ** ($depth * 8)) - $busrouterSize));
            $busrouter->offset = (2 ** ($depth * 8)) - $busrouterSize + 1;
            $components[] = $busrouter;
        };
        $this->components = $components;
    }

    public function read(int $byte) : int {
        foreach($this->components as &$component) {
            if($component->isByteBelongToComponent($byte)) {
                return $component->component->read($byte - $component->offset);
            }
        }
        return 0;
    }
    public function write(int $byte, int $value) : void {
        foreach($this->components as &$component) {
            if($component->isByteBelongToComponent($byte)) {
                $component->component->write($byte - $component->offset, $value);
                return;
            }
        }
    }
}

class LoadedComponent {
    private int $bytes;
    public readonly Component $component;
    public int $offset;
    
    public function __construct(Component $component) {
        if(!isset($component->bytes)) throw new Exception("Can't load component " . get_class($component) . ": bytes property is null");
        $this->bytes = $component->bytes;
        $this->component = $component;
    }
    
    public function getBytes() : int {
        return $this->bytes;
    }

    public function isByteBelongToComponent(int $byte) : bool {
        return $byte >= $this->offset & $byte < $this->offset + $this->getBytes();
    }
}

class Processor {
    private Components $components;
    private Opcodes $opcodes;
    private Registers $registers;

    public function __construct(int $depth, Components &$components) {
        $this->components = $components;
        $this->registers = new Registers($depth);
        $this->opcodes = new Opcodes($this->components, $this->registers, $depth);
    }

    public function tick() : void {
        $programCounter = $this->registers->read(0);
        $this->opcodes->execute($this->components->read($programCounter));
        $this->registers->write(0, $this->registers->read(0) + 1);
    }
}

class Registers {
    private array $registers;

    public function __construct(int &$depth) {
        $registers = [];
        for($i = 1; $i <= 8; $i++) {
            $registers[] = new Register($depth);
        }
        $this->registers = $registers;
    }

    public function read(int $register) : int {
        return $this->registers[$register]->read();
    }
    public function write(int $register, int $value) : void {
        $this->registers[$register]->write($value);
    }
}

class Register {
    private int $value;
    private int $depth;

    public function __construct(int &$depth) {
        $this->value = 0;
        $this->depth = &$depth;
    }

    public function read() : int {
        return $this->value;
    }
    public function write(int $value) : void {
        $this->value = $value & intval((2 ** ($this->depth * 8)) - 1);
    }
}

class Opcodes {
    private array $opcodes;

    public function __construct(Components &$components, Registers &$registers, int &$depth) {
        $opcodes = [];

        $opcodes[] = new MOV($components, $registers, $depth);
        $opcodes[] = new NAND($components, $registers, $depth);
        $opcodes[] = new BIT($components, $registers, $depth);
        $opcodes[] = new JMPC($components, $registers, $depth);

        $this->opcodes = $opcodes;
    }

    public function execute(int $byte) : void {
        $this->opcodes[$byte >> 6]->execute($byte & 0b111111);
    }
}

class busrouter extends Component {
    private string $contents;
    public readonly int $bytes;

    private function number2bytes(int $number, int $bytes) : string {
        $string = '';
        $shift = 255 << (8 * ($bytes - 1));
        for($i = 0; $i < $bytes; $i++) {
            $string .= chr(($number & $shift) >> (8 * ($bytes - 1 - $i))); // might be rewritten
            $shift >>= 8;
        }
        return $string;
    }

    public function __construct(int $depth, array $components, int $offset) {
        $contents = '';
        $pointers = '';
        foreach($components as &$component) {
            $contents .= $this->number2bytes($component->offset, $depth);
            $contents .= $this->number2bytes($component->getBytes(), $depth);
            $contents .= get_class($component->component) . "\0";
        }
        $contents .= $pointers;
        $this->bytes = strlen($contents);
        $this->contents = $contents;
    }

    public function read(int $byte) : int {
        return $this->contents[$byte];
    }
    public function write(int $byte, int $value) : void {

    }
}

new Debugger;
(new Machine)->run();